﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Text;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Utils;
using gov.va.med.vbecs.Common.Log;

namespace gov.va.med.VBECS.Communication.Protocols
{
    /// <summary>
    /// Protocol to communicate with Vista side
    /// This protocol doesn't preserve message type information.
    /// </summary>
    public class MessagesProtocolEot : IProtocol
    {
        /// <summary>
        /// This MemoryStream object is used to collect receiving bytes to build messages.
        /// </summary>
        private MemoryStream _receiveMemoryStream = new MemoryStream();
        // This is special logger for communication tracing
        private readonly ILogger _traceLogger =
            LogManager.Instance().LoggerLocator.GetLogger("CommunicationTrace");
        // Message serializer
        private readonly IMessageSerializer _sr;
        // Messages factory
        private readonly IMessageFactory _messagesFactory = new RawDataMessageFactory();

        /// <summary>
        /// Default constructor
        /// </summary>
        public MessagesProtocolEot()
        {
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="sr">Message serializer</param>
        /// <param name="messagesFactory">Messages factory deserializes messages from bytes</param>
        public MessagesProtocolEot(IMessageSerializer sr, IMessageFactory messagesFactory)
        {
            _sr = sr;
            _messagesFactory = messagesFactory;
        }

        /// <summary>
        /// Converts message content to bytes array
        /// </summary>
        /// <param name="theMsg"></param>
        /// <returns>Bytes</returns>
        /// <exception cref="ArgumentNullException">Message object</exception>
        public byte[] GetBytes(IMessage theMsg)
        {
            if (theMsg == null)
                throw new ArgumentNullException("theMsg");

            var msgBytes = theMsg.GetBytes();
            _traceLogger.Info(
                string.Format("Protocol type: {0}, out message:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, Encoding.ASCII.GetString(msgBytes))
                );

            // Create a byte array including 1 byte (EOT) and serialized message content
            var result = new byte[msgBytes.Length + 1];
            Array.Copy(msgBytes, 0, result, 0, msgBytes.Length);
            // Write EOT byte: http://en.wikipedia.org/wiki/End_of_transmission_character
            result[msgBytes.Length] = Convert.ToByte(4);
            // Return serialized message by this protocol

            if (_sr != null)
                _sr.SerializeObject("Messages\\Responce_eot_" + Guid.NewGuid(), new RawDataMessage(result));

            return result;
        }


        /// <summary>
        /// Builds message object from bytes array
        /// </summary>
        /// <param name="theBytes">Bytes array to build message</param>
        /// <returns></returns>
        public IEnumerable<IMessage> BuildMessages(byte[] theBytes)
        {
            //Create a list to collect messages
            var msgList = new List<IMessage>();

            //Return false to wait more bytes from remote application.
            if (theBytes.Length < 1)
                return msgList;
            
            //Check is the last byte is EOT byte
            if( theBytes[theBytes.Length-1] == Convert.ToByte(4))
            {
                //We reached the end of transmission
                //Write all received bytes to the _receiveMemoryStream
                _receiveMemoryStream.Write(theBytes, 0, theBytes.Length-1);
                //Check if it is ping message
                Guid outGuid;
                if (Guid.TryParse(new string(Encoding.UTF8.GetChars(_receiveMemoryStream.ToArray())), out outGuid))
                {
                    msgList.Add(new PingMessage(outGuid.ToString()));
                    _traceLogger.Debug(
                        string.Format("Protocol type: {0}, ping message:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, Encoding.UTF8.GetString(theBytes))
                        );
                    if (_sr != null)
                        _sr.SerializeObject("Messages\\Ping_eot_" + Guid.NewGuid(), new RawDataMessage(theBytes));
                }
                else
                {
                    var message = _messagesFactory.CreateMessage(_receiveMemoryStream.ToArray());
                    msgList.Add(message);
                    _traceLogger.Info(
                        string.Format("Protocol type: {0}, in-whole message:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, Encoding.UTF8.GetString(message.GetBytes()))
                        );
                    if (_sr != null)
                        _sr.SerializeObject("Messages\\Whole_eot_" + Guid.NewGuid(), message);
                }
                // Clear the stream and wait for new message (CR 3342)
                _receiveMemoryStream = new MemoryStream();
            }
            else
            {
                //Return and wait for more bytes
                //Note that current position is placed at the end of receiving stream. So next bytes will be appended to the end.
                _receiveMemoryStream.Write(theBytes, 0, theBytes.Length);
                _traceLogger.Debug(
                    string.Format("Protocol type: {0}, in-part message:\r\n{1}", MethodBase.GetCurrentMethod().DeclaringType, Encoding.UTF8.GetString(theBytes))
                    );
                if (_sr != null)
                    _sr.SerializeObject("Messages\\Part_eot_" + Guid.NewGuid(), new RawDataMessage(theBytes));
            }

            return msgList;
        }

        /// <summary>
        /// Resets internal receive stream
        /// </summary>
        public void Reset()
        {
            if (_receiveMemoryStream.Length > 0)
            {
                _receiveMemoryStream = new MemoryStream();
            }
        }
    }
}
